'use client';

import { createSlice, createAsyncThunk } from '@reduxjs/toolkit';
import apiRequest from '../../InterceptorAxios/axios';

// Fetch Shortcuts (Now includes hidden shortcuts)
export const fetchShortcuts = createAsyncThunk(
  'shortcuts/fetchShortcuts',
  async ({ companyId, userId }, thunkAPI) => {
    try {
      
      
      const response = await apiRequest(
        `/api/shortcuts/${companyId}?userId=${userId}`,  
        'GET'
      );
      
      
      return {
        shortcuts: response.shortcuts || [],
    
        publicShortcuts: response.publicShortcuts || [],
        privateShortcuts: response.privateShortcuts || [],
        hiddenShortcuts: response.hiddenShortcuts || [],
      
      };
    } catch (error) {
      return thunkAPI.rejectWithValue(error.response?.data || 'Failed to fetch shortcuts');
    }
  }
);

// Add Shortcut
export const addShortcut = createAsyncThunk(
  'shortcuts/addShortcut',
  async ({ userId, companyId, shortcut, shortcutType }, thunkAPI) => {
    try {
      
      const response = await apiRequest(
        `/api/shortcuts/create`,
        'POST',
        { shortcutName: shortcut,userId, companyId, shortcutType }
      );
      
      return {
        shortcuts: response.shortcuts,
        shortcutdetails: response.shortcutdetails,
      };
    } catch (error) {
      return thunkAPI.rejectWithValue(error.response?.data || 'Failed to add shortcut');
    }
  }
);

// Delete Shortcut
export const deleteShortcut = createAsyncThunk(
  'shortcuts/deleteShortcut',
  async ({ userId, companyId, shortcutId }, thunkAPI) => {
    try {
      await apiRequest(
        `/api/shortcuts/delete`,
        'DELETE',
        { shortcutId, companyId }
      );
      return shortcutId; 
    } catch (error) {
      return thunkAPI.rejectWithValue(error.response?.data || 'Failed to delete shortcut');
    }
  }
);

// Hide Shortcut
export const hideShortcut = createAsyncThunk(
  'shortcuts/hideShortcut',
  async ({ userId, shortcutId }, thunkAPI) => {
    try {
      await apiRequest(
        `/api/shortcuts/hide`,
        'PUT',
        { shortcutId,userId }
      );
      return shortcutId;
    } catch (error) {
      return thunkAPI.rejectWithValue(error.response?.data || 'Failed to hide shortcut');
    }
  }
);

// Unhide Shortcut
export const unhideShortcut = createAsyncThunk(
  'shortcuts/unhideShortcut',
  async ({ userId, shortcutId }, thunkAPI) => {
    try {
      await apiRequest(
        `/api/shortcuts/unhide`,
        'PUT',
        { shortcutId,userId }
      );
      return shortcutId;
    } catch (error) {
      return thunkAPI.rejectWithValue(error.response?.data || 'Failed to unhide shortcut');
    }
  }
);

// Slice
const shortcutsSlice = createSlice({
  name: 'shortcuts',
  initialState: {
    shortcuts: [],
    shortcutdetails: [],
    status: 'idle',
    error: null,
  },
  reducers: {},
  extraReducers: (builder) => {
    builder
      .addCase(fetchShortcuts.fulfilled, (state, action) => {
        
        state.shortcuts = 
          action.payload.shortcuts
        
        state.shortcutdetails = [...action.payload.privateShortcuts, ...action.payload.publicShortcuts ];
        
        
      })
      .addCase(addShortcut.fulfilled, (state, action) => {
        state.shortcuts.push(action.payload.shortcuts);
        state.shortcutdetails.push(action.payload.shortcutdetails);
      })
      .addCase(deleteShortcut.fulfilled, (state, action) => {
        const payloadShortcutId = action.payload;
        state.shortcutdetails = state.shortcutdetails.filter((s) => s.id !== payloadShortcutId);
        state.shortcuts = state.shortcuts.filter((s) => s.id !== payloadShortcutId);
      })
      .addCase(hideShortcut.fulfilled, (state, action) => {
     
          
          // Remove the hidden shortcut from the shortcuts array
          state.shortcuts = state.shortcuts.filter((shortcut) => shortcut.id !== action.payload);
          // Find and update the hidden shortcut in the shortcutdetails array
          const hiddenShortcut = state.shortcutdetails.find((s) => s.id === action.payload);
          
          hiddenShortcut.is_hidden = 1;
          if (hiddenShortcut) {
          }
          
      })
      .addCase(unhideShortcut.fulfilled, (state, action) => {
        // Find the unhidden shortcut in the shortcutdetails array
        const unhiddenShortcut = state.shortcutdetails.find((s) => s.id === action.payload);
      
        if (unhiddenShortcut) {
          // Create a simplified object to add to the shortcuts array
          const simplifiedShortcut = {
            text: unhiddenShortcut.shortcuts,  // Assuming `text` is the text you want to display
            type: unhiddenShortcut.shortcut_type === 1 ? 'public' : 'private',  // Shortcut type check
            id: unhiddenShortcut.id
          };
      
          // Update the is_hidden status to 0 (unhide it)
          unhiddenShortcut.is_hidden = 0;
      
          // Add the simplified shortcut back to the shortcuts array
          state.shortcuts.push(simplifiedShortcut);
        }
      });
     
  },
});

export default shortcutsSlice.reducer;
