import axios from 'axios';
import { deleteCookie, getCookie, setCookie } from 'cookies-next';
import { disconnectSocket, initializeSocket } from '../socketManager';
import CryptoJS from 'crypto-js';
import handlelogout from '@/app/utils/handlelogout';

// const SECRET_KEY = CryptoJS.SHA256(process.env.NEXT_PUBLIC_SECRET_KEY).toString(
//   CryptoJS.enc.Hex
// );
const SECRET_KEY = CryptoJS.SHA256(process.env.NEXT_PUBLIC_SECRET_KEY);


// Function to check if response data is encrypted
const isEncrypted = (data) => {
  return (
    typeof data === "object" &&
    data.hasOwnProperty("e") &&
    data.hasOwnProperty("iv")
  );
};


// Function to decrypt AES-256-GCM encrypted data using crypto-js

// Function to decrypt AES-256-CBC encrypted data
// Function to decrypt AES-256-CBC encrypted data
const decryptData = ({ e, iv }) => {
  try {
    const key = SECRET_KEY; // Already parsed correctly
    const ivBytes = CryptoJS.enc.Base64.parse(iv);
    const encryptedBytes = CryptoJS.enc.Base64.parse(e);

    const decrypted = CryptoJS.AES.decrypt(
      { ciphertext: encryptedBytes },
      key,
      { iv: ivBytes, mode: CryptoJS.mode.CBC, padding: CryptoJS.pad.Pkcs7 }
    );

    const decryptedText = decrypted.toString(CryptoJS.enc.Utf8);
    if (!decryptedText) throw new Error("Decryption resulted in an empty string");

    return JSON.parse(decryptedText);
  } catch (error) {
    console.error("Decryption failed:", error.message);
    return null;
  }
};


// Create a custom Axios instance
const axiosInstance = axios.create({
  baseURL: process.env.NEXT_PUBLIC_API_URL,
  headers: {
    'Content-Type': 'application/json',
  },
  timeout: 10000,
});

// Request interceptor to add token to headers
axiosInstance.interceptors.request.use(
  (config) => {
    const token = getCookie('token') || '';
    if (token) {
      config.headers['Authorization'] = `Bearer ${token}`;
    }
    return config;
  },
  (error) => {
    console.error('Request Error:', error.message);
    return Promise.reject(error);
  }
);

// Response interceptor to decrypt response data if necessary
axiosInstance.interceptors.response.use(
  (response) => {
    if (response.data && isEncrypted(response.data)) {
      const decryptedData = decryptData(response.data);
      return { ...response, data: decryptedData || response.data };
    }
    return response;
  },
  async (error) => {
    const originalRequest = error.config;

    if (error.response && error.response.status === 401 && !originalRequest._retry) {
      originalRequest._retry = true;
      try {
        const { data } = await axios.post(`${process.env.NEXT_PUBLIC_API_URL}/api/auth/refresh-token`, {
          refreshToken: getCookie('refreshToken'),
        });

        setCookie('token', data.accessToken);
        setCookie('refreshToken', data.refreshToken);
        originalRequest.headers['Authorization'] = `Bearer ${data.accessToken}`;

        disconnectSocket();
        initializeSocket(data.accessToken, data.refreshToken);

        return axiosInstance(originalRequest);
      } catch (err) {
        handlelogout();
        return Promise.reject(err);
      }
    }
    return Promise.reject(error);
  }
);

// Reusable API request function
const apiRequest = async (url, method = 'GET', data = null, additionalHeaders = {}) => {
  try {
    const isFormData = data instanceof FormData;

    const response = await axiosInstance({
      url,
      method,
      headers: {
        ...additionalHeaders,
        ...(isFormData ? { 'Content-Type': 'multipart/form-data' } : { 'Content-Type': 'application/json' }),
      },
      data: isFormData ? data : data,
    });

    return response.data;
  } catch (error) {
    if (error.type) {
      if (error.type === 'client') throw new Error(error.message);
      if (error.type === 'server') throw new Error('Server Error: ' + error.message);
      if (error.type === 'network') throw new Error(error.message);
      throw new Error('An unexpected error occurred.');
    } else {
      throw new Error('An error occurred. Please try again.');
    }
  }
};

export default apiRequest;